/* bessel.cpp */

#include "fp.h"


#define maxSizeExponent 29

static bool P(fp& res, const fp& n, const fp& x);
static bool Q(fp& res, const fp& n, const fp& x);
static bool R(fp& res, const fp& n, const fp& x);
static bool S(fp& res, const fp& n, const fp& x);
static void seriesBesselJ(fp& res, const fp& n, const fp& x, fp& lastGamma,
							const short& codeGamma);
static void seriesBesselI(fp& res, const fp& n, const fp& x, fp& lastGamma,
							const short& codeGamma);
static bool besselIByAsym(fp& res, const fp& n, const fp& x);
static bool besselKByDiff(fp& res, const fp& n, const fp& x);
static bool besselKByAsym(fp& res, const fp& n, const fp& x);
static bool besselKBySineRatio(fp& res, const fp& n, const fp& x);

//extern bool isPiGood;
//extern fp myPi;

//long Lg2(const fp& x);

// asymptotic series returns true if ending is good, false if ending is bad
// use for J and Y
static bool P(fp& z, const fp& n, const fp& x)
{
	fp		mu, temp1, temp2;
	INT32	i;
	
	mu = 4*n*n;
	
	fp	t2, s, s1, t, t1;

    s = 0;
    t = 1;
    t2 = -1/(64*x*x);
    t1 = t;

    for (i = 1; ; i+=2)
    {
       add(s1, s, t);
       if (s == s1) break;
       s = s1;
       mul(t, t, t2);
       div(t, t, i*(i+1));
       sub(temp1, mu, (2*i-1)*(2*i-1));
       sub(temp2, mu, (2*i+1)*(2*i+1));
       mul(t, t, temp1*temp2);
       if(abs(t)>abs(t1))
       {
       		z = s;
       		return false;
       }
       else
       		t1 = t;
    }
    
    z = s;
    return true;
    
}/* P */

// asymptotic series returns true if ending is good, false if ending is bad
// use for J and Y
static bool Q(fp& z, const fp& n, const fp& x)
{
	fp		mu, temp1, temp2;
	INT32	i;
	
	mu = 4*n*n;
	
	fp	t2, s, s1, t, t1;

    s = 0;
    t = (mu-1)/(8*x);
    t2 = -1/(64*x*x);
    t1 = t;

    for (i = 1; ; i+=2)
    {
       add(s1, s, t);
       if (s == s1) break;
       s = s1;
       mul(t, t, t2);
       div(t, t, (i+1)*(i+2));
       sub(temp1, mu, (2*i+1)*(2*i+1));
       sub(temp2, mu, (2*i+3)*(2*i+3));
       mul(t, t, temp1*temp2);
       if(abs(t)>abs(t1))
       {
       		z = s;
       		return false;
       }
       else
       		t1 = t;
    }
    
    z = s;
    return true;
    
}/* Q */

// asymptotic series returns true if ending is good, false if ending is bad
// use for I
static bool R(fp& z, const fp& n, const fp& x)
{
	fp		mu;
	INT32	i;
	
	mu = 4*n*n;
	
	fp	t2, s, s1, t, t1;

    s = 0;
    t = 1;
    t2 = -1/(8*x);
    t1 = t;

    for (i = 1; ; i++)
    {
       add(s1, s, t);
       if (s == s1) break;
       s = s1;
       mul(t, t, t2);
       div(t, t, i);
       mul(t, t, mu - (2*i-1)*(2*i-1));
       if(abs(t)>abs(t1))
       {
       		z = s;
       		return false;
       }
       else
       		t1 = t;
    }
    
    z = s;
    return true;
    
}/* R */

// asymptotic series returns true if ending is good, false if ending is bad
// use for K
static bool S(fp& z, const fp& n, const fp& x)
{
	fp		mu;
	INT32	i;
	
	mu = 4*n*n;
	
	fp	t2, s, s1, t, t1;

    s = 0;
    t = 1;
    t2 = 1/(8*x);
    t1 = t;

    for (i = 1; ; i++)
    {
       add(s1, s, t);
       if (s == s1) break;
       s = s1;
       mul(t, t, t2);
       div(t, t, i);
       mul(t, t, mu - (2*i-1)*(2*i-1));
       if(abs(t)>abs(t1))
       {
       		z = s;
       		return false;
       }
       else
       		t1 = t;
    }
    
    z = s;
    return true;
    
}/* S */


static bool asymptoticBesselJ(fp& z, const fp& n, const fp& x)
{
	fp		chi, one, two, four, oneFourth;
	fp		temp1, temp2;
	bool	isPrecisionGood, tempBool;
	
	
	one = 1.;
	two = 2.;
	four = 4.;
	oneFourth = one/four;
	 
	chi = x - (n/2 + oneFourth)*Pi();
	isPrecisionGood = P(temp1, n, x);
	tempBool = Q(temp2, n, x);
	if(!tempBool)
		isPrecisionGood = false;
		
	z = sqrt(two/(x*Pi()))*(temp1*cos(chi) - temp2*sin(chi));
	
	return isPrecisionGood;

}/* asymptoticBesselJ */


// use seriesBesselJ when abs(n) > x*x/4
// don't use for n a negative integer
static void seriesBesselJ(fp& z, const fp& n, const fp& x, fp& lastGamma,
							const short& codeGamma)
{
   INT32	i;
   INT32	p = getBlockPrec();
	
   setBlockPrec(p + NumBits(blockBits*p)/blockBits + 1);
	
	//fp::SetPrecision(p + NumBits(p) + 10);
   
   fp	temp;
   fp	t2, s, s1, t;

   s = 0;
   t = 1;
   t2 = -(x*x/4);

   for (i = 1; ; i++)
   {
      add(s1, s, t);
      if (s == s1) break;
      s = s1;
      mul(t, t, t2);
      div(t, t, i*(n+i));
   }
   
   if(n.i.n==0)
   		temp = 1;
   else
   		temp = pow(x/2, n);
   		
   z = temp*s;
   
   switch(codeGamma)
   {
   		case 0:
   			Gamma(temp, n+1);
   			lastGamma = temp;
            break;
    	case 1:
    		temp = n*lastGamma;
            break;
    	default: break;
   }
   
   setBlockPrec(p);
	
   z = z/temp;
   
}/* seriesBesselJ */


// use seriesBesselI when abs(n) > x*x/4
static void seriesBesselI(fp& z, const fp& n, const fp& x, fp& lastGamma,
							const short& codeGamma)
{
   INT32	p = getBlockPrec();
	
   setBlockPrec(p + NumBits(blockBits*p)/blockBits + 1);
   INT32	i;
   fp	temp;
   fp	t2, s, s1, t;

   s = 0;
   t = 1;
   t2 = x*x/4;

   for (i = 1; ; i++)
   {
      add(s1, s, t);
      if (s == s1) break;
      s = s1;
      mul(t, t, t2);
      div(t, t, i*(n+i));
   }
   
   if(n.i.n==0)
   		temp = 1;
   else
   		temp = pow(x/2, n);
   		
   z = temp*s;
   
   switch(codeGamma)
   {
   		case 0:
   			Gamma(temp, n+1);
   			lastGamma = temp;
            break;
    	case 1:
    		temp = n*lastGamma;
            break;
    	default: break;
   }
   
   setBlockPrec(p);
   z = z/temp;
   
}/* seriesBesselI */


bool besselJ(fp& z, const fp& n, const fp& x)
{
	bool		isPrecisionGood, tempBool, isNegInt;
	fp			res, lastGamma, chi, two=2., temp, temp1, temp2;
	fp			fracPart;
	short		codeGamma;
	mb			kmb;
	INT32		i, k, nInt;
	fp			nn;
	
	nn  = n; // may need to negate nn
	
	INT32	p = getBlockPrec();
	
	setBlockPrec(p + NumBits(blockBits*p)/blockBits + 1);
	
	isPrecisionGood = true;
	
	if(x.i.n==0)
	{
		// never call besselJ if x=0 and n<0.
		if(nn.i.n==0)
			z = 1.;
		else
			z = 0;
		
		setBlockPrec(p);
			
		return isPrecisionGood;
	}
	
	// is n a negative integer?
	isNegInt = equate(nInt, nn);
	if(isNegInt)
	{
		// is an integer, but is it negative?
		if(nInt>0)
			isNegInt = false;
		else
			nn = -nn; // make nn positive
	}

	if(abs(x)<40) // was 25
	{
		codeGamma = 0;
		seriesBesselJ(res, nn, x, lastGamma, codeGamma);
		if(isNegInt)
		{
			if(2*(nInt/2) != nInt)
				res = -res;
		}
		z = res;
		setBlockPrec(p);
		
		return isPrecisionGood;
	}
	
	// now abs(x) > 40
	
	if(nn.i.n>=0)
	{
		// nn = k + fracPart
		equate(kmb, nn);
		// test for long overflow
		equate(k, kmb);
		sub(fracPart, nn, k);
		
		setBlockPrec(p + (INT32)sqrt(k)/blockBits);
		//fp::SetPrecision(fp::precision() + (INT32)sqrt(k));
		
		if(k==0 || k==1)
		{
			isPrecisionGood = asymptoticBesselJ(res, nn, x);
			if(isNegInt)
			{
				if(2*(nInt/2) != nInt)
					res = -res;
			}
			z = res;
			setBlockPrec(p);
			return isPrecisionGood;
		}
		
		if(nn<=abs(x))
		{
			// do recursion in direction of increasing n
			isPrecisionGood = asymptoticBesselJ(temp1, fracPart, x);
			tempBool = asymptoticBesselJ(temp2, fracPart+1, x);
			if(!tempBool)
				isPrecisionGood = false;
		
			// we know k>=2
			for(i=0;i<k-1;i++)
			{
				fracPart++;
				temp = two*fracPart*temp2/x - temp1;
				temp1 = temp2;
				temp2 = temp;
			}
			res = temp2;
			if(isNegInt)
			{
				if(2*(nInt/2) != nInt)
					res = -res;
			}
			z = res;
			setBlockPrec(p);
			
			return isPrecisionGood;
		}
		else
		{
			// now n>abs(x) (like n=100, x=41, x*x=1681)
			// do recursion in direction of decreasing n
			// but what should the starting order be?
			// Numerical Recipes says start larger than the desired n by an additive
			// amount of order [constant x n]^(1/2) where the square root of the constant
			// is, very roughly, the number of significant figures of accuracy.
			// or we could try the series with increased precision
			// problem occurs when J is close to zero
			// for n=100 and 108.83 < x < 108.84
			// Should increase precision as J->0 (up to certain point)
			// also look at n=1100 x=100
			
			setBlockPrec(p + 3);
			//fp::SetPrecision(fp::precision() + 100); // this helps for n=100, x=99
			codeGamma = 0;
			seriesBesselJ(res, nn, x, lastGamma, codeGamma);
			z = res;
			setBlockPrec(p);
			
			return isPrecisionGood;
		}
	}
	else
	{
		equate(kmb, abs(nn));
		// test for long overflow
		equate(k, kmb);
		sub(fracPart, abs(nn), k);
		fracPart = 1 - fracPart;
		k++;
		
		setBlockPrec(p + (INT32)sqrt(k)/blockBits);
		
		isPrecisionGood = asymptoticBesselJ(temp1, fracPart, x);
		tempBool = asymptoticBesselJ(temp2, fracPart+1, x);
		if(!tempBool)
			isPrecisionGood = false;
			
		for(i=0;i<k;i++)
		{
			temp = two*fracPart*temp1/x - temp2;
			temp2 = temp1;
			temp1 = temp;
			fracPart--;
		}
		res = temp1;
		if(isNegInt)
		{
			if(2*(nInt/2) != nInt)
				res = -res;
		}
		z = res;
		setBlockPrec(p);
		
		return isPrecisionGood;
	}
	
	setBlockPrec(p);
	
	return isPrecisionGood;
	
}/* besselJ */


// http://www.efunda.com/math/bessel/bessel.cfm
// must have x>0
static bool besselYIntegerOrder(fp& z, INT32 n, const fp& x)
{
	fp		fact1, fact2, fact3, saveFact3;
	fp		temp, term, nn, x2;
	fp		sum1, sum2, one=1;
	bool	isPrecisionGood;
	INT32	m;
	
	if(!x.i.n)
		return false;
	
	INT32 p = getBlockPrec();
	setBlockPrec(p + NumBits(blockBits*p)/blockBits + 1);
	
	// fact1
	
	nn = n;
	isPrecisionGood = besselJ(temp, nn, x);
	fact1 = 2*temp*(log(x/2) + EulerGamma());
	
	// fact2
	
	fact2 = 0;
	
	if(n)
	{
		x2 = x/2;
		pow(term, x2, -nn);
		term = term*factorial(nn-1);
		x2 = x2*x2; // (x/2)^2
		
		for(m=1;m<=n;++m)
		{
			fact2 = fact2 + term;
			if(m==n)
				break;
			term = term*x2/(m*(nn-m));
		}
	}
	
	// fact3
	
	sum1 = 0;
	sum2 = 0;
	fact3 = 0;
	
	for(m=1;m<=n;m++)
	{
		sum2 = sum2 + one/m;
	}
	
	x2 = x/2;
	pow(term, x2, nn);
	term = -term/factorial(nn);
	saveFact3 = 1;  // first fact3 can only be <=0
	
	x2 = -x2*x2;
	
	for(m=1;;m++)
	{
		fact3 = fact3 + (sum1 + sum2)*term;
		if(fact3==saveFact3)
			break;
		saveFact3 = fact3;
		sum1 = sum1 + one/m;
		sum2 = sum2 + one/(nn+m);
		term = term*x2/(m*(nn+m));
	}
	
	z = fact1 - fact2 + fact3;
	
	z = z/Pi();
	
	setBlockPrec(p);
	
	return isPrecisionGood;
	
}/* besselYIntegerOrder */


static bool besselYnonIntegerOrder(fp& z, const fp& n, const fp& x)
{
	bool		itIsLong;
	fp			res, temp1, temp2, temp3, nAbs;
	bool		isPrecisionGood, tempBool;
	INT32		longNumber, longNumber2;
	bool		isHalfOddInt;
	fp			small;
	
	isPrecisionGood = true;
	
	// the closer n is to an integer the more precision should be used
	//itIsLong = myIsLong(n, longNumber);
	itIsLong = equate(longNumber, n);
	if(itIsLong)
		return false;
	
	INT32	p = getBlockPrec();
	INT32	nLong, addPrec;
	fp		nFrac1, nFrac2;
	
	equate(nLong, n);
	nFrac1 = n - nLong;
	nFrac2 = 1 - nFrac1;
	temp1 = abs(Lg2(nFrac1));
	temp2 = abs(Lg2(nFrac2));
	if(temp1>=temp2)
		equate(addPrec, temp1);
	else
		equate(addPrec, temp2);
	
	setBlockPrec(p + addPrec/blockBits + 1);
	
	// is n half an odd integer?
	isHalfOddInt = equate(longNumber2, 2*n);
	if(isHalfOddInt)
	{
		// we have an integer, is it odd?
		if(2*(longNumber2/2)==longNumber2)
			isHalfOddInt = false;
	}
	
	if(!isHalfOddInt)
	{
		isPrecisionGood = besselJ(temp1, n, x);
		tempBool = besselJ(temp2, -n, x);
		if(!tempBool)
			isPrecisionGood = false;
		temp3 = n*Pi();
		z = (temp1*cos(temp3) - temp2)/sin(temp3);
		setBlockPrec(p);
		return isPrecisionGood;
	}
	else
	{
		// do have half odd integer; is it even or odd integer + 1/2?
		bool		isEven;
		INT32		temp;
		nAbs = abs(n);
		nAbs = nAbs - .5;
		equate(temp, nAbs);
		if(temp == 2*(temp/2))
			isEven = true;
		else
			isEven = false;
		isPrecisionGood = besselJ(temp2, -n, x);
		res = -temp2;
		if(!isEven)
			res = -res;
		if(n.i.n<0)
			res = -res;
		z = res;
		setBlockPrec(p);
		return isPrecisionGood;
	}
	
}/* besselYnonIntegerOrder */


bool besselY(fp& z, const fp& n, const fp& x)
{
	fp			res, nn;
	INT32		nInt;
	bool		isNegInt, isPrecisionGood;
	
	nn = n;
	
	isNegInt = equate(nInt, nn);
	
	if(isNegInt)
	{
		// is an integer, but is it negative?
		if(nInt>0)
			isNegInt = false;
		else
			nInt = -nInt; // make nInt positive
		
		isPrecisionGood = besselYIntegerOrder(res, nInt, x);
		
		if(isNegInt)
		{
			if(2*(nInt/2) != nInt)
				res = -res;
		}
		
		z = res;
		
		return isPrecisionGood;
	}
	
	
	isPrecisionGood = besselYnonIntegerOrder(res, nn, x);
	
	z = res;
	
	return isPrecisionGood;
	
}/* besselY */


bool besselI(fp& z, const fp& n, const fp& x)
{
	fp		res, temp1, two;
	bool	isPrecisionGood;
	fp		cutOff, lastGamma;
	short	codeGamma;
	
	INT32 p = getBlockPrec();
	cutOff = .5*p*blockBits + 5*abs(n); // also as abs(n) goes up so should cutOff

	setBlockPrec(p + NumBits(blockBits*p)/blockBits + 1);
   //	fp::SetPrecision(p + NumBits(p) + 10); // was p + NumBits(p) + 10, then 3*p
	two = 2.;
	isPrecisionGood = true;
	
	if(abs(x)<=cutOff)
	{
		codeGamma = 0;
		seriesBesselI(res, n, x, lastGamma, codeGamma);
		z = res;
		setBlockPrec(p);
		return isPrecisionGood;
	}
	
	// now abs(x) > cutOff so we use asymptotic series R
	
	isPrecisionGood = R(temp1, n, x);
	
	z = temp1*exp(x)/sqrt(two*x*Pi());
	
	setBlockPrec(p);
	
	return isPrecisionGood;
	
}/* besselI */


// this is faster
static bool besselKIntegerOrder(fp& z, INT32 n, const fp& x)
{
	fp		fact1, fact2, fact3, saveFact3;
	fp		temp, term, nn, x2;
	fp		sum1, sum2, one=1;
	bool	isPrecisionGood;
	INT32	m;
	
	z = 0;
	
	if(!x.i.n)
		return false;
	
	INT32 p = getBlockPrec();
	setBlockPrec(p + NumBits(blockBits*p)/blockBits + 2);
	
	// fact1
	
	nn = n;
	isPrecisionGood = besselI(temp, nn, x);
	fact1 = temp*(log(x/2) + EulerGamma());
	if(2*(n/2)==n)
		fact1 = -fact1;
	
	// fact2
	
	fact2 = 0;
	
	if(n)
	{
		x2 = x/2;
		pow(term, x2, -nn);
		term = term*factorial(nn-1)/2;
		x2 =-x2*x2; // (x/2)^2
		
		for(m=1;m<=n;++m)
		{
			fact2 = fact2 + term;
			if(m==n)
				break;
			term = term*x2/(m*(nn-m));
		}
	}
	
	
	// fact3
	
	sum1 = 0;
	sum2 = 0;
	fact3 = 0;
	
	for(m=1;m<=n;++m)
	{
		sum2 = sum2 + one/m;
	}
	
	x2 = x/2;
	pow(term, x2, nn);
	term = -term/(2*factorial(nn));
	if(2*(n/2)==n)
		term = -term;
	saveFact3 = 1 + (sum1 + sum2)*term; // no match on first iteration below
	
	x2 = x2*x2;
	
	for(m=1;;++m)
	{
		fact3 = fact3 + (sum1 + sum2)*term;
		if(fact3==saveFact3)
			break;
		saveFact3 = fact3;
		sum1 = sum1 + one/m;
		sum2 = sum2 + one/(nn+m);
		term = term*x2/(m*(nn+m));
	}
	
	z = fact1 + fact2 + fact3;
	
	setBlockPrec(p);
	
	return isPrecisionGood;
	
}/* besselKIntegerOrder */


// this works for both integer and non-integer order, but is slow for integer order compared to above routine
// 
static bool besselKanyOrder(fp& z, const fp& n, const fp& x)
{
	INT32	nDiff, i;
	fp		res, nBigfp, temp1, temp2, temp3, two=2.;
	bool	isPrecisionGood, tempBool;
	fp		xCutOff;
	
	// we use upward recurrence relation
	INT32 p = getBlockPrec();
	setBlockPrec(p + NumBits(blockBits*p)/blockBits + 1);
	
	isPrecisionGood = true;
	tempBool = true;
	
	xCutOff = .5*p*blockBits + 5;

	equate(nDiff, abs(n));
	
	if(n.i.n>=0 && n<two)
	{
		if(abs(x)<=xCutOff)
			besselKBySineRatio(res, n, x);
		else
			isPrecisionGood = besselKByAsym(res, n, x);
		
		z = res;
		setBlockPrec(p);
		return isPrecisionGood;
	}
	
	// now n>=2
	nDiff-=1;
	nBigfp = n - nDiff;
	if(abs(x)<=xCutOff)
	{
		besselKBySineRatio(temp1, nBigfp-1, x);
		besselKBySineRatio(temp2, nBigfp, x);
	}
	else
	{
		isPrecisionGood = besselKByAsym(temp1, nBigfp-1, x);
		tempBool = besselKByAsym(temp2, nBigfp, x);
	}
	if(!tempBool)
		isPrecisionGood = false;
	for(i=0;i<nDiff;i++)
	{
		temp3 = two*nBigfp*temp2/x + temp1;
		temp1 = temp2;
		temp2 = temp3;
		nBigfp+=1;
	}
	z = temp3;
	setBlockPrec(p);
	return isPrecisionGood;
	
}/* besselKanyOrder */


bool besselK(fp& z, const fp& n, const fp& x)
{
	fp			res, nn;
	INT32		nInt;
	bool		isInt, isPrecisionGood;
	
	nn = abs(n);
	isInt = equate(nInt, nn);
	
	if(isInt)
		isPrecisionGood = besselKIntegerOrder(res, nInt, x);
	else
		isPrecisionGood = besselKanyOrder(res, nn, x);
	
	z = res;
	
	return isPrecisionGood;
	
}/* besselK */


static bool besselIByAsym(fp& z, const fp& n, const fp& x)
{
	fp			temp;
	bool		isPrecisionGood;

	isPrecisionGood = R(temp, n, x);
	
	z = temp*exp(x)/sqrt(2*x*Pi());
	
	// fp::SetPrecision(p);
	
	return isPrecisionGood;
	
}/* besselIByAsym */


static bool besselKByAsym(fp& z, const fp& n, const fp& x)
{
	fp			temp;
	bool		isPrecisionGood;
	
	isPrecisionGood = S(temp, n, x);
	
	z = temp*exp(-x)*sqrt(Pi()/(2*x));
	
	return isPrecisionGood;
	
}/* besselKByAsym */


static bool besselKBySineRatio(fp& z, const fp& n, const fp& x)
{
	bool		isPrecisionGood, tempBool, itIsLong;
	INT32		longNumber;
	fp			temp1, temp2, temp;
	
	INT32 p = getBlockPrec();
	
	//itIsLong = myIsLong(n, longNumber);
	itIsLong = equate(longNumber, n);
	
	if(!itIsLong)
	{
		INT32	nLong, addPrec;
		fp		nFrac1, nFrac2;
		// the closer n is to an integer the more precision should be used
		equate(nLong, n);
		nFrac1 = n - nLong;
		nFrac2 = 1 - nFrac1;
		temp1 = abs(Lg2(nFrac1));
		temp2 = abs(Lg2(nFrac2));
		if(temp1>=temp2)
			equate(addPrec, temp1);
		else
			equate(addPrec, temp2);
		
		setBlockPrec(p + addPrec/blockBits + 1);
		isPrecisionGood = besselI(temp1, -n, x);
		tempBool = besselI(temp2, n, x);
		if(!tempBool)
			isPrecisionGood = false;
		z = (Pi()/2)*(temp1-temp2)/sin(n*Pi());
		setBlockPrec(p);
		return isPrecisionGood;
	}
	
	// now n is an integer so double precision and make n slightly larger
	
	fp		dn, two, nNew, pMinus;
	two = 2;
	setBlockPrec(3*p);  // was 2*p then 1.2*p
	pMinus = -blockBits*p/1.9; // seems optimum but doesn't scale with precision properly
	dn = pow(two, pMinus);
	nNew = n + dn;
	isPrecisionGood = besselI(temp1, -nNew, x);
	tempBool = besselI(temp2, nNew, x);
	if(!tempBool)
		isPrecisionGood = false;
	// try an average
	temp = (temp1 - temp2)/sin(nNew*Pi());
	nNew = n - dn;
	tempBool = besselI(temp1, -nNew, x);
	if(!tempBool)
		isPrecisionGood = false;
	tempBool = besselI(temp2, nNew, x);
	if(!tempBool)
		isPrecisionGood = false;
	temp = (temp + (temp1-temp2)/sin(nNew*Pi()))/2;
	z = (Pi()/2)*temp;
	setBlockPrec(p);
	return isPrecisionGood;
	
}/* besselKBySineRatio */

bool besselJzero(fp& z, const fp& n, const fp& x)
{
	fp		index, indexNext, y;
	fp		J, Jnext;
	fp		denom;
	fp		eps, ten=10.;
	INT32	p, m, mMax=100; // nMax=1000 takes about 21 seconds for normal precision; adjust for precision? no
	
	p = getBlockPrec();
	setBlockPrec(p+1);
	power(eps, ten, -getDecPrec());
	y = x;
	index = n;
	indexNext = index + 1;
	m = 0;
	
	while(true)
	{
		besselJ(J, index, y);
		if(abs(J)<eps)
			break;
		besselJ(Jnext, indexNext, y);
		denom = J*index/y - Jnext;
		if(denom.i.n==0)
		{
			setBlockPrec(p);
			return false;
		}
			
		y = y - J/denom;
		
		if(++m>mMax || y.i.n<=0)
		{
			z = y;
			setBlockPrec(p);
			return false;
		}
	}
	
	z = y;
	setBlockPrec(p);
	return true;
	
}/* besselJzero */


bool besselYzero(fp& z, const fp& n, const fp& x)
{
	fp		index, indexNext, y;
	fp		Y, Ynext;
	fp		denom;
	fp		eps, ten=10.;
	INT32	p, m, mMax=100; // nMax=1000 takes about 21 seconds for normal precision; adjust for precision? no
	
	p = getBlockPrec();
	setBlockPrec(p+1);
	power(eps, ten, -getDecPrec());
	y = x;
	index = n;
	indexNext = index + 1.;
	m = 0;
	
	while(true)
	{
		besselY(Y, index, y);
		if(abs(Y)<eps)
			break;
		besselY(Ynext, indexNext, y);
		denom = Y*index/y - Ynext;
		if(denom.i.n==0)
		{
			setBlockPrec(p);
			return false;
		}
		y = y - Y/denom;
		
		if(++m>mMax || y.i.n<=0)
		{
			z = y;
			setBlockPrec(p);
			return false;
		}
	}
	
	z = y;
	setBlockPrec(p);
	return true;
	
}/* besselYzero */


bool sphericalBesselJ(fp& z, fp& n, fp& x)
{
	fp			res, one=1;
	bool		isPrecisionGood;
	
	if(x.i.n<0)
		return false;
	
	if(!x.i.n && !n.i.n)
	{
		z = one;
		return true;
	}
	
	if(!x.i.n)
	{
		z = 0;
		return true;
	}
	
	isPrecisionGood = besselJ(res, n+.5, x);
	
	z = res*sqrt(Pi()/(2*x));
	
	return isPrecisionGood;
	
}/* sphericalBesselJ */


bool sphericalBesselY(fp& z, fp& n, fp& x)
{
	fp			res;
	bool		isPrecisionGood;
	
	if(x.i.n<=0)
		return false;
	
	isPrecisionGood = besselY(res, n+.5, x);
	
	z = res*sqrt(Pi()/(2*x));
	
	return isPrecisionGood;
	
}/* sphericalBesselY */
